/*
 * Copyright (C) 2014 The Android Open Source Project
 * Copyright (c) 2005, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.util;

import java.io.IOException;
import java.util.Iterator;
import java.util.stream.Stream;

/**
 * A facility to load implementations of a service.
 *
 * <p> A <i>service</i> is a well-known interface or class for which zero, one,
 * or many service providers exist. A <i>service provider</i> (or just
 * <i>provider</i>) is a class that implements or subclasses the well-known
 * interface or class. A {@code ServiceLoader} is an object that locates and
 * loads service providers deployed in the run time environment at a time of an
 * application's choosing. Application code refers only to the service, not to
 * service providers, and is assumed to be capable of choosing between multiple
 * service providers (based on the functionality they expose through the service),
 * and handling the possibility that no service providers are located.
 *
 * <h2> Obtaining a service loader </h2>
 *
 * <p> An application obtains a service loader for a given service by invoking
 * one of the static {@code load} methods of {@code ServiceLoader}. <!--If the
 * application is a module, then its module declaration must have a <i>uses</i>
 * directive that specifies the service; this helps to locate providers and ensure
 * they will execute reliably. In addition, if the application module does not
 * contain the service, then its module declaration must have a <i>requires</i>
 * directive that specifies the module which exports the service. It is strongly
 * recommended that the application module does <b>not</b> require modules which
 * contain providers of the service.--/>
 *
 * <p> A service loader can be used to locate and instantiate providers of the
 * service by means of the {@link #iterator() iterator} method. {@code ServiceLoader}
 * also defines the {@link #stream() stream} method to obtain a stream of providers
 * that can be inspected and filtered without instantiating them.
 *
 * <p> As an example, suppose the service is {@code com.example.CodecFactory}, an
 * interface that defines methods for producing encoders and decoders:
 *
 * <pre>{@code
 *     package com.example;
 *     public interface CodecFactory {
 *         Encoder getEncoder(String encodingName);
 *         Decoder getDecoder(String encodingName);
 *     }
 * }</pre>
 *
 * <p> The following code obtains a service loader for the {@code CodecFactory}
 * service, then uses its iterator (created automatically by the enhanced-for
 * loop) to yield instances of the service providers that are located:
 *
 * <pre>{@code
 *     ServiceLoader<CodecFactory> loader = ServiceLoader.load(CodecFactory.class);
 *     for (CodecFactory factory : loader) {
 *         Encoder enc = factory.getEncoder("PNG");
 *         if (enc != null)
 *             ... use enc to encode a PNG file
 *             break;
 *         }
 * }</pre>
 *
 * <p> Sometimes an application may wish to inspect a service provider before
 * instantiating it, in order to determine if an instance of that service
 * provider would be useful. For example, a service provider for {@code
 * CodecFactory} that is capable of producing a "PNG" encoder may be annotated
 * with {@code @PNG}. The following code uses service loader's {@code stream}
 * method to yield instances of {@code Provider<CodecFactory>} in contrast to
 * how the iterator yields instances of {@code CodecFactory}:
 * <pre>{@code
 *     ServiceLoader<CodecFactory> loader = ServiceLoader.load(CodecFactory.class);
 *     Set<CodecFactory> pngFactories = loader
 *            .stream()                                              // Note a below
 *            .filter(p -> p.type().isAnnotationPresent(PNG.class))  // Note b
 *            .map(Provider::get)                                    // Note c
 *            .collect(Collectors.toSet());
 * }</pre>
 * <ol type="a">
 *   <li> A stream of {@code Provider<CodecFactory>} objects </li>
 *   <li> {@code p.type()} yields a {@code Class<CodecFactory>} </li>
 *   <li> {@code get()} yields an instance of {@code CodecFactory} </li>
 * </ol>
 *
 * <h2> Designing services </h2>
 *
 * <p> A service is a single type, usually an interface or abstract class. A
 * concrete class can be used, but this is not recommended. The type may have
 * any accessibility. The methods of a service are highly domain-specific, so
 * this API specification cannot give concrete advice about their form or
 * function. However, there are two general guidelines:
 * <ol>
 *   <li><p> A service should declare as many methods as needed to allow service
 *   providers to communicate their domain-specific properties and other
 *   quality-of-implementation factors. An application which obtains a service
 *   loader for the service may then invoke these methods on each instance of
 *   a service provider, in order to choose the best provider for the
 *   application. </p></li>
 *   <li><p> A service should express whether its service providers are intended
 *   to be direct implementations of the service or to be an indirection
 *   mechanism such as a "proxy" or a "factory". Service providers tend to be
 *   indirection mechanisms when domain-specific objects are relatively
 *   expensive to instantiate; in this case, the service should be designed
 *   so that service providers are abstractions which create the "real"
 *   implementation on demand. For example, the {@code CodecFactory} service
 *   expresses through its name that its service providers are factories
 *   for codecs, rather than codecs themselves, because it may be expensive
 *   or complicated to produce certain codecs. </p></li>
 * </ol>
 *
 * <h2> <a id="developing-service-providers">Developing service providers</a> </h2>
 *
 * <p> A service provider is a single type, usually a concrete class. An
 * interface or abstract class is permitted because it may declare a static
 * provider method, discussed later. The type must be public and must not be
 * an inner class.
 *
 * <p> A service provider and its supporting code may be developed in a module,
 * which is then deployed on the application module path or in a modular
 * image. Alternatively, a service provider and its supporting code may be
 * packaged as a JAR file and deployed on the application class path. The
 * advantage of developing a service provider in a module is that the provider
 * can be fully encapsulated to hide all details of its implementation.
 *
 * <p> An application that obtains a service loader for a given service is
 * indifferent to whether providers of the service are deployed in modules or
 * packaged as JAR files. The application instantiates service providers via
 * the service loader's iterator, or via {@link java.util.ServiceLoader.Provider Provider} objects in
 * the service loader's stream, without knowledge of the service providers'
 * locations.
 *
 * <h2> Deploying service providers on the class path </h2>
 *
 * A service provider that is packaged as a JAR file for the class path is
 * identified by placing a <i>provider-configuration file</i> in the resource
 * directory {@code META-INF/services}. The name of the provider-configuration
 * file is the fully qualified binary name of the service. The provider-configuration
 * file contains a list of fully qualified binary names of service providers, one
 * per line.
 *
 * <p> For example, suppose the service provider
 * {@code com.example.impl.StandardCodecs} is packaged in a JAR file for the
 * class path. The JAR file will contain a provider-configuration file named:
 *
 * <blockquote>{@code
 *     META-INF/services/com.example.CodecFactory
 * }</blockquote>
 *
 * that contains the line:
 *
 * <blockquote>{@code
 *     com.example.impl.StandardCodecs # Standard codecs
 * }</blockquote>
 *
 * <p><a id="format">The provider-configuration file must be encoded in UTF-8. </a>
 * Space and tab characters surrounding each service provider's name, as well as
 * blank lines, are ignored. The comment character is {@code '#'}
 * ({@code U+0023} <span style="font-size:smaller;">NUMBER SIGN</span>);
 * on each line all characters following the first comment character are ignored.
 * If a service provider class name is listed more than once in a
 * provider-configuration file then the duplicate is ignored. If a service
 * provider class is named in more than one configuration file then the duplicate
 * is ignored.
 *
 * <p> A service provider that is mentioned in a provider-configuration file may
 * be located in the same JAR file as the provider-configuration file or in a
 * different JAR file. The service provider must be visible from the class loader
 * that is initially queried to locate the provider-configuration file; this is
 * not necessarily the class loader which ultimately locates the
 * provider-configuration file.
 *
 * <h2> Timing of provider discovery </h2>
 *
 * <p> Service providers are loaded and instantiated lazily, that is, on demand.
 * A service loader maintains a cache of the providers that have been loaded so
 * far. Each invocation of the {@code iterator} method returns an {@code Iterator}
 * that first yields all of the elements cached from previous iteration, in
 * instantiation order, and then lazily locates and instantiates any remaining
 * providers, adding each one to the cache in turn. Similarly, each invocation
 * of the stream method returns a {@code Stream} that first processes all
 * providers loaded by previous stream operations, in load order, and then lazily
 * locates any remaining providers. Caches are cleared via the {@link #reload
 * reload} method.
 *
 * <h2> <a id="errors">Errors</a> </h2>
 *
 * <p> When using the service loader's {@code iterator}, the {@link java.util.Iterator#hasNext() hasNext} and {@link java.util.Iterator#next() next} methods will
 * fail with {@link java.util.ServiceConfigurationError ServiceConfigurationError} if an error occurs locating,
 * loading or instantiating a service provider. When processing the service
 * loader's stream then {@code ServiceConfigurationError} may be thrown by any
 * method that causes a service provider to be located or loaded.
 *
 * <p> When loading or instantiating a service provider in a module, {@code
 * ServiceConfigurationError} can be thrown for the following reasons:
 *
 * <p> When reading a provider-configuration file, or loading or instantiating
 * a provider class named in a provider-configuration file, then {@code
 * ServiceConfigurationError} can be thrown for the following reasons:
 *
 * <ul>
 *
 *   <li> The format of the provider-configuration file violates the <a
 *   href="ServiceLoader.html#format">format</a> specified above; </li>
 *
 *   <li> An {@link java.io.IOException IOException} occurs while reading the
 *   provider-configuration file; </li>
 *
 *   <li> A service provider cannot be loaded; </li>
 *
 *   <li> A service provider is not assignable to the service's interface or
 *   class, or does not define a provider constructor, or cannot be
 *   instantiated. </li>
 *
 * </ul>
 *
 * <h2> Concurrency </h2>
 *
 * <p> Instances of this class are not safe for use by multiple concurrent
 * threads.
 *
 * <h3> Null handling </h3>
 *
 * <p> Unless otherwise specified, passing a {@code null} argument to any
 * method in this class will cause a {@link java.lang.NullPointerException NullPointerException} to be thrown.
 *
 * @param  <S>
 *         The type of the service to be loaded by this loader
 *
 * @author Mark Reinhold
 * @since 1.6
 * @revised 9
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class ServiceLoader<S> implements java.lang.Iterable<S> {

ServiceLoader() { throw new RuntimeException("Stub!"); }

/**
 * Returns an iterator to lazily load and instantiate the available
 * providers of this loader's service.
 *
 * <p> To achieve laziness the actual work of locating and instantiating
 * providers is done by the iterator itself. Its {@link java.util.Iterator#hasNext  hasNext} and {@link java.util.Iterator#next next} methods can therefore throw a
 * {@link java.util.ServiceConfigurationError ServiceConfigurationError} for any of the reasons specified in
 * the <a href="#errors">Errors</a> section above. To write robust code it
 * is only necessary to catch {@code ServiceConfigurationError} when using
 * the iterator. If an error is thrown then subsequent invocations of the
 * iterator will make a best effort to locate and instantiate the next
 * available provider, but in general such recovery cannot be guaranteed.
 *
 * <p> Caching: The iterator returned by this method first yields all of
 * the elements of the provider cache, in the order that they were loaded.
 * It then lazily loads and instantiates any remaining service providers,
 * adding each one to the cache in turn. If this loader's provider caches are
 * cleared by invoking the {@link #reload() reload} method then existing
 * iterators for this service loader should be discarded.
 * The {@code  hasNext} and {@code next} methods of the iterator throw {@link
 * java.util.ConcurrentModificationException ConcurrentModificationException}
 * if used after the provider cache has been cleared.
 *
 * <p> The iterator returned by this method does not support removal.
 * Invoking its {@link java.util.Iterator#remove() remove} method will
 * cause an {@link java.lang.UnsupportedOperationException UnsupportedOperationException} to be thrown.
 *
 * @apiNote Throwing an error in these cases may seem extreme.  The rationale
 * for this behavior is that a malformed provider-configuration file, like a
 * malformed class file, indicates a serious problem with the way the Java
 * virtual machine is configured or is being used.  As such it is preferable
 * to throw an error rather than try to recover or, even worse, fail silently.
 *
 * @return  An iterator that lazily loads providers for this loader's
 *          service
 *
 * @revised 9
 */

public java.util.Iterator<S> iterator() { throw new RuntimeException("Stub!"); }

/**
 * Returns a stream to lazily load available providers of this loader's
 * service. The stream elements are of type {@link java.util.ServiceLoader.Provider Provider}, the
 * {@code Provider}'s {@link java.util.ServiceLoader.Provider#get() get} method must be invoked to
 * get or instantiate the provider.
 *
 * <p> To achieve laziness the actual work of locating providers is done
 * when processing the stream. If a service provider cannot be loaded for any
 * of the reasons specified in the <a href="#errors">Errors</a> section
 * above then {@link java.util.ServiceConfigurationError ServiceConfigurationError} is thrown by whatever method
 * caused the service provider to be loaded. </p>
 *
 * <p> Caching: When processing the stream then providers that were previously
 * loaded by stream operations are processed first, in load order. It then
 * lazily loads any remaining service providers. If this loader's provider
 * caches are cleared by invoking the {@link #reload() reload} method then
 * existing streams for this service loader should be discarded. The returned
 * stream's source {@link java.util.Spliterator spliterator} is <em>fail-fast</em> and
 * will throw {@link java.util.ConcurrentModificationException ConcurrentModificationException} if the provider cache
 * has been cleared. </p>
 *
 * <p> The following examples demonstrate usage. The first example creates
 * a stream of {@code CodecFactory} objects, the second example is the same
 * except that it sorts the providers by provider class name (and so locate
 * all providers).
 * <pre>{@code
 *    Stream<CodecFactory> providers = ServiceLoader.load(CodecFactory.class)
 *            .stream()
 *            .map(Provider::get);
 *
 *    Stream<CodecFactory> providers = ServiceLoader.load(CodecFactory.class)
 *            .stream()
 *            .sorted(Comparator.comparing(p -> p.type().getName()))
 *            .map(Provider::get);
 * }</pre>
 *
 * @return  A stream that lazily loads providers for this loader's service
 *
 * @since 9
 */

public java.util.stream.Stream<java.util.ServiceLoader.Provider<S>> stream() { throw new RuntimeException("Stub!"); }

/**
 * Creates a new service loader for the given service. The service loader
 * uses the given class loader as the starting point to locate service
 * providers for the service.
 *
 * @apiNote If the class path of the class loader includes remote network
 * URLs then those URLs may be dereferenced in the process of searching for
 * provider-configuration files.
 *
 * <p> This activity is normal, although it may cause puzzling entries to be
 * created in web-server logs.  If a web server is not configured correctly,
 * however, then this activity may cause the provider-loading algorithm to fail
 * spuriously.
 *
 * <p> A web server should return an HTTP 404 (Not Found) response when a
 * requested resource does not exist.  Sometimes, however, web servers are
 * erroneously configured to return an HTTP 200 (OK) response along with a
 * helpful HTML error page in such cases.  This will cause a {@link java.util.ServiceConfigurationError ServiceConfigurationError} to be thrown when this class attempts to parse
 * the HTML page as a provider-configuration file.  The best solution to this
 * problem is to fix the misconfigured web server to return the correct
 * response code (HTTP 404) along with the HTML error page.
 *
 * @param  <S> the class of the service type
 *
 * @param  service
 *         The interface or abstract class representing the service
 *
 * @param  loader
 *         The class loader to be used to load provider-configuration files
 *         and provider classes, or {@code null} if the system class
 *         loader (or, failing that, the bootstrap class loader) is to be
 *         used
 *
 * @return A new service loader
 *
 * @throws java.util.ServiceConfigurationError
 *         if the service type is not accessible to the caller <!-- or the
 *         caller is in an explicit module and its module descriptor does
 *         not declare that it uses {@code service} --/>
 *
 * @revised 9
 */

public static <S> java.util.ServiceLoader<S> load(java.lang.Class<S> service, java.lang.ClassLoader loader) { throw new RuntimeException("Stub!"); }

/**
 * Creates a new service loader for the given service type, using the
 * current thread's {@linkplain java.lang.Thread#getContextClassLoader
 * context class loader}.
 *
 * <p> An invocation of this convenience method of the form
 * <pre>{@code
 *     ServiceLoader.load(service)
 * }</pre>
 *
 * is equivalent to
 *
 * <pre>{@code
 *     ServiceLoader.load(service, Thread.currentThread().getContextClassLoader())
 * }</pre>
 *
 * @apiNote Service loader objects obtained with this method should not be
 * cached VM-wide. For example, different applications in the same VM may
 * have different thread context class loaders. A lookup by one application
 * may locate a service provider that is only visible via its thread
 * context class loader and so is not suitable to be located by the other
 * application. Memory leaks can also arise. A thread local may be suited
 * to some applications.
 *
 * @param  <S> the class of the service type
 *
 * @param  service
 *         The interface or abstract class representing the service
 *
 * @return A new service loader
 *
 * @throws java.util.ServiceConfigurationError
 *         if the service type is not accessible to the caller
 *
 * @revised 9
 */

public static <S> java.util.ServiceLoader<S> load(java.lang.Class<S> service) { throw new RuntimeException("Stub!"); }

/**
 * Creates a new service loader for the given service type, using the
 * extension class loader.
 *
 * <p>This convenience method simply locates the extension class loader,
 * call it {@code extClassLoader}, and then returns
 *
 * <pre>{@code
 *     ServiceLoader.load(service, extClassLoader)
 * }</pre>
 *
 * <p> This method is intended for use when only installed providers are
 * desired.  The resulting service will only find and load providers that
 * have been installed into the current Java virtual machine.
 *
 * @param  <S> the class of the service type
 *
 * @param  service
 *         The interface or abstract class representing the service
 *
 * @return A new service loader
 *
 * @throws java.util.ServiceConfigurationError
 *         if the service type is not accessible to the caller
 *
 * @revised 9
 */

public static <S> java.util.ServiceLoader<S> loadInstalled(java.lang.Class<S> service) { throw new RuntimeException("Stub!"); }

/**
 * Load the first available service provider of this loader's service. This
 * convenience method is equivalent to invoking the {@link #iterator()
 * iterator()} method and obtaining the first element. It therefore
 * returns the first element from the provider cache if possible, it
 * otherwise attempts to load and instantiate the first provider.
 *
 * <p> The following example loads the first available service provider. If
 * no service providers are located then it uses a default implementation.
 * <pre>{@code
 *    CodecFactory factory = ServiceLoader.load(CodecFactory.class)
 *                                        .findFirst()
 *                                        .orElse(DEFAULT_CODECSET_FACTORY);
 * }</pre>
 * @return The first service provider or empty {@code Optional} if no
 *         service providers are located
 *
 * @throws java.util.ServiceConfigurationError
 *         If a provider class cannot be loaded for any of the reasons
 *         specified in the <a href="#errors">Errors</a> section above.
 *
 * @since 9
 */

public java.util.Optional<S> findFirst() { throw new RuntimeException("Stub!"); }

/**
 * Clear this loader's provider cache so that all providers will be
 * reloaded.
 *
 * <p> After invoking this method, subsequent invocations of the {@link
 * #iterator() iterator} or {@link #stream() stream} methods will lazily
 * locate providers (and instantiate in the case of {@code iterator})
 * from scratch, just as is done by a newly-created service loader.
 *
 * <p> This method is intended for use in situations in which new service
 * providers can be installed into a running Java virtual machine.
 */

public void reload() { throw new RuntimeException("Stub!"); }

/**
 * Returns a string describing this service.
 *
 * @return  A descriptive string
 */

public java.lang.String toString() { throw new RuntimeException("Stub!"); }
/**
 * Represents a service provider located by {@code ServiceLoader}.
 *
 * <p> When using a loader's {@link java.util.ServiceLoader#stream() stream()} method
 * then the elements are of type {@code Provider}. This allows processing
 * to select or filter on the provider class without instantiating the
 * provider. </p>
 *
 * @param  <S> The service type
 * @since 9
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface Provider<S> extends java.util.function.Supplier<S> {

/**
 * Returns the provider type. There is no guarantee that this type is
 * accessible or that it has a public no-args constructor. The {@link
 * #get() get()} method should be used to obtain the provider instance.
 *
 * <p> When a module declares that the provider class is created by a
 * provider factory then this method returns the return type of its
 * public static "{@code provider()}" method.
 *
 * @return The provider type
 */

public java.lang.Class<? extends S> type();

/**
 * Returns an instance of the provider.
 *
 * @return An instance of the provider.
 *
 * @throws java.util.ServiceConfigurationError
 *         If the service provider cannot be instantiated, or in the
 *         case of a provider factory, the public static
 *         "{@code provider()}" method returns {@code null} or throws
 *         an error or exception. The {@code ServiceConfigurationError}
 *         will carry an appropriate cause where possible.
 */

public S get();
}

}

