/*
 * Copyright (C) 2010 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.nfc;


/**
 * Represents an immutable NDEF Message.
 * <p>
 * NDEF (NFC Data Exchange Format) is a light-weight binary format,
 * used to encapsulate typed data. It is specified by the NFC Forum,
 * for transmission and storage with NFC, however it is transport agnostic.
 * <p>
 * NDEF defines messages and records. An NDEF Record contains
 * typed data, such as MIME-type media, a URI, or a custom
 * application payload. An NDEF Message is a container for
 * one or more NDEF Records.
 * <p>
 * When an Android device receives an NDEF Message
 * (for example by reading an NFC tag) it processes it through
 * a dispatch mechanism to determine an activity to launch.
 * The type of the <em>first</em> record in the message has
 * special importance for message dispatch, so design this record
 * carefully.
 * <p>
 * Use {@link #NdefMessage(byte[])} to construct an NDEF Message from
 * binary data, or {@link #NdefMessage(android.nfc.NdefRecord[])} to
 * construct from one or more {@link android.nfc.NdefRecord NdefRecord}s.
 * <p class="note">
 * {@link android.nfc.NdefMessage NdefMessage} and {@link android.nfc.NdefRecord NdefRecord} implementations are
 * always available, even on Android devices that do not have NFC hardware.
 * <p class="note">
 * {@link android.nfc.NdefRecord NdefRecord}s are intended to be immutable (and thread-safe),
 * however they may contain mutable fields. So take care not to modify
 * mutable fields passed into constructors, or modify mutable fields
 * obtained by getter methods, unless such modification is explicitly
 * marked as safe.
 *
 * @see android.nfc.NfcAdapter#ACTION_NDEF_DISCOVERED
 * @see android.nfc.NdefRecord
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class NdefMessage implements android.os.Parcelable {

/**
 * Construct an NDEF Message by parsing raw bytes.<p>
 * Strict validation of the NDEF binary structure is performed:
 * there must be at least one record, every record flag must
 * be correct, and the total length of the message must match
 * the length of the input data.<p>
 * This parser can handle chunked records, and converts them
 * into logical {@link android.nfc.NdefRecord NdefRecord}s within the message.<p>
 * Once the input data has been parsed to one or more logical
 * records, basic validation of the tnf, type, id, and payload fields
 * of each record is performed, as per the documentation on
 * on {@link android.nfc.NdefRecord#NdefRecord(short,byte[],byte[],byte[]) NdefRecord#NdefRecord(short, byte[], byte[], byte[])}<p>
 * If either strict validation of the binary format fails, or
 * basic validation during record construction fails, a
 * {@link android.nfc.FormatException FormatException} is thrown<p>
 * Deep inspection of the type, id and payload fields of
 * each record is not performed, so it is possible to parse input
 * that has a valid binary format and confirms to the basic
 * validation requirements of
 * {@link android.nfc.NdefRecord#NdefRecord(short,byte[],byte[],byte[]) NdefRecord#NdefRecord(short, byte[], byte[], byte[])},
 * but fails more strict requirements as specified by the
 * NFC Forum.
 *
 * <p class="note">
 * It is safe to re-use the data byte array after construction:
 * this constructor will make an internal copy of all necessary fields.
 *
 * @param data raw bytes to parse
 * @throws android.nfc.FormatException if the data cannot be parsed
 */

public NdefMessage(byte[] data) throws android.nfc.FormatException { throw new RuntimeException("Stub!"); }

/**
 * Construct an NDEF Message from one or more NDEF Records.
 *
 * @param record first record (mandatory)
 * @param records additional records (optional)
 */

public NdefMessage(android.nfc.NdefRecord record, android.nfc.NdefRecord... records) { throw new RuntimeException("Stub!"); }

/**
 * Construct an NDEF Message from one or more NDEF Records.
 *
 * @param records one or more records
 */

public NdefMessage(android.nfc.NdefRecord[] records) { throw new RuntimeException("Stub!"); }

/**
 * Get the NDEF Records inside this NDEF Message.<p>
 * An {@link android.nfc.NdefMessage NdefMessage} always has one or more NDEF Records: so the
 * following code to retrieve the first record is always safe
 * (no need to check for null or array length >= 1):
 * <pre>
 * NdefRecord firstRecord = ndefMessage.getRecords()[0];
 * </pre>
 *
 * @return array of one or more NDEF records.
 */

public android.nfc.NdefRecord[] getRecords() { throw new RuntimeException("Stub!"); }

/**
 * Return the length of this NDEF Message if it is written to a byte array
 * with {@link #toByteArray}.<p>
 * An NDEF Message can be formatted to bytes in different ways
 * depending on chunking, SR, and ID flags, so the length returned
 * by this method may not be equal to the length of the original
 * byte array used to construct this NDEF Message. However it will
 * always be equal to the length of the byte array produced by
 * {@link #toByteArray}.
 *
 * @return length of this NDEF Message when written to bytes with {@link #toByteArray}
 * @see #toByteArray
 */

public int getByteArrayLength() { throw new RuntimeException("Stub!"); }

/**
 * Return this NDEF Message as raw bytes.<p>
 * The NDEF Message is formatted as per the NDEF 1.0 specification,
 * and the byte array is suitable for network transmission or storage
 * in an NFC Forum NDEF compatible tag.<p>
 * This method will not chunk any records, and will always use the
 * short record (SR) format and omit the identifier field when possible.
 *
 * @return NDEF Message in binary format
 * @see #getByteArrayLength()
 */

public byte[] toByteArray() { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

public void writeToParcel(android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the specified NDEF Message contains
 * identical NDEF Records.

 * @param obj This value may be {@code null}.
 */

public boolean equals(@android.annotation.Nullable java.lang.Object obj) { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.nfc.NdefMessage> CREATOR;
static { CREATOR = null; }
}

