// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.


package android.net.http;

import java.nio.ByteBuffer;
import java.util.concurrent.Executor;

/**
 * Class for bidirectional sending and receiving of data over HTTP/2 or QUIC connections. Created by
 * {@link android.net.http.BidirectionalStream.Builder Builder}.
 *
 * Note: There are ordering restrictions on methods of {@link android.net.http.BidirectionalStream BidirectionalStream};
 * please see individual methods for description of restrictions.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract class BidirectionalStream {

public BidirectionalStream() { throw new RuntimeException("Stub!"); }

/**
 * See {@link android.net.http.BidirectionalStream.Builder#setHttpMethod(java.lang.String) BidirectionalStream.Builder#setHttpMethod(String)}.
 */

@android.annotation.NonNull
public abstract java.lang.String getHttpMethod();

/**
 * See {@link android.net.http.BidirectionalStream.Builder#setTrafficStatsTag(int) BidirectionalStream.Builder#setTrafficStatsTag(int)}
 */

public abstract boolean hasTrafficStatsTag();

/**
 * See {@link android.net.http.BidirectionalStream.Builder#setTrafficStatsTag(int) BidirectionalStream.Builder#setTrafficStatsTag(int)}
 */

public abstract int getTrafficStatsTag();

/**
 * See {@link android.net.http.BidirectionalStream.Builder#setTrafficStatsUid(int) BidirectionalStream.Builder#setTrafficStatsUid(int)}
 */

public abstract boolean hasTrafficStatsUid();

/**
 * See {@link android.net.http.BidirectionalStream.Builder#setTrafficStatsUid(int) BidirectionalStream.Builder#setTrafficStatsUid(int)}
 */

public abstract int getTrafficStatsUid();

/**
 * See {@link android.net.http.BidirectionalStream.Builder#addHeader(java.lang.String,java.lang.String) Builder#addHeader(String, String)}
 */

@android.annotation.NonNull
public abstract android.net.http.HeaderBlock getHeaders();

/**
 * See {@link android.net.http.BidirectionalStream.Builder#setPriority(int) Builder#setPriority(int)}

 * @return Value is {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_IDLE}, {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_LOWEST}, {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_LOW}, {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_MEDIUM}, or {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_HIGHEST}
 */

public abstract int getPriority();

/**
 * See {@link android.net.http.BidirectionalStream.Builder#setDelayRequestHeadersUntilFirstFlushEnabled(boolean) Builder#setDelayRequestHeadersUntilFirstFlushEnabled(boolean)}
 */

public abstract boolean isDelayRequestHeadersUntilFirstFlushEnabled();

/**
 * Starts the stream, all callbacks go to the {@code callback} argument passed to {@link android.net.http.BidirectionalStream.Builder BidirectionalStream.Builder}'s constructor. Should only be called once.
 */

public abstract void start();

/**
 * Reads data from the stream into the provided buffer. Can only be called at most once in
 * response to each invocation of the {@link android.net.http.BidirectionalStream.Callback#onStreamReady onStreamReady()}/ {@link android.net.http.BidirectionalStream.Callback#onResponseHeadersReceived onResponseHeadersReceived()} and {@link android.net.http.BidirectionalStream.Callback#onReadCompleted onReadCompleted()} methods of the {@link android.net.http.BidirectionalStream.Callback Callback}. Each call will
 * result in an invocation of one of the {@link android.net.http.BidirectionalStream.Callback Callback}'s {@link android.net.http.BidirectionalStream.Callback#onReadCompleted onReadCompleted()} method if data is read, or its {@link android.net.http.BidirectionalStream.Callback#onFailed onFailed()} method if there's an error.
 *
 * An attempt to read data into {@code buffer} starting at {@code buffer.position()} is begun.
 * At most {@code buffer.remaining()} bytes are read. {@code buffer.position()} is updated upon
 * invocation of {@link android.net.http.BidirectionalStream.Callback#onReadCompleted onReadCompleted()} to indicate how much data
 * was read.
 *
 * @param buffer the {@link java.nio.ByteBuffer ByteBuffer} to read data into. Must be a direct ByteBuffer. The
 * embedder must not read or modify buffer's position, limit, or data between its position and
 * limit until {@link android.net.http.BidirectionalStream.Callback#onReadCompleted onReadCompleted()}, {@link android.net.http.BidirectionalStream.Callback#onCanceled  onCanceled()}, or {@link android.net.http.BidirectionalStream.Callback#onFailed onFailed()} are invoked.
 */

public abstract void read(@android.annotation.NonNull java.nio.ByteBuffer buffer);

/**
 * Attempts to write data from the provided buffer into the stream. If auto flush is disabled,
 * data will be sent only after {@link #flush flush()} is called. Each call will result in an
 * invocation of one of the {@link android.net.http.BidirectionalStream.Callback Callback}'s {@link android.net.http.BidirectionalStream.Callback#onWriteCompleted  onWriteCompleted()} method if data is sent, or its {@link android.net.http.BidirectionalStream.Callback#onFailed onFailed()}
 * method if there's an error.
 *
 * An attempt to write data from {@code buffer} starting at {@code buffer.position()} is begun.
 * {@code buffer.remaining()} bytes will be written. {@link android.net.http.BidirectionalStream.Callback#onWriteCompleted  onWriteCompleted()} will be invoked only when the full ByteBuffer is written.
 *
 * @param buffer the {@link java.nio.ByteBuffer ByteBuffer} to write data from. Must be a direct ByteBuffer. The
 * embedder must not read or modify buffer's position, limit, or data between its position and
 * limit until {@link android.net.http.BidirectionalStream.Callback#onWriteCompleted onWriteCompleted()}, {@link android.net.http.BidirectionalStream.Callback#onCanceled  onCanceled()}, or {@link android.net.http.BidirectionalStream.Callback#onFailed onFailed()} are invoked. Can be empty when {@code
 * endOfStream} is {@code true}.
 * @param endOfStream if {@code true}, then {@code buffer} is the last buffer to be written, and
 * once written, stream is closed from the client side, resulting in half-closed stream or a
 * fully closed stream if the remote side has already closed.
 */

public abstract void write(@android.annotation.NonNull java.nio.ByteBuffer buffer, boolean endOfStream);

/**
 * Flushes pending writes. This method should not be invoked before {@link android.net.http.BidirectionalStream.Callback#onStreamReady onStreamReady()}. For previously delayed {@link #write write()}s, a
 * corresponding {@link android.net.http.BidirectionalStream.Callback#onWriteCompleted onWriteCompleted()} will be invoked when the
 * buffer is sent.
 */

public abstract void flush();

/**
 * Cancels the stream. Can be called at any time after {@link #start}. {@link android.net.http.BidirectionalStream.Callback#onCanceled onCanceled()} will be invoked when cancelation is complete and no further
 * callback methods will be invoked. If the stream has completed or has not started, calling
 * {@code cancel()} has no effect and {@code onCanceled()} will not be invoked. If the {@link java.util.concurrent.Executor Executor} passed in during
 * {@code BidirectionalStream} construction runs tasks on a single thread, and {@code cancel()}
 * is called on that thread, no listener methods (besides {@code onCanceled()}) will be invoked
 * after
 * {@code cancel()} is called. Otherwise, at most one callback method may be invoked after
 * {@code cancel()} has completed.
 */

public abstract void cancel();

/**
 * Returns {@code true} if the stream was successfully started and is now done (succeeded,
 * canceled, or failed).
 *
 * @return {@code true} if the stream was successfully started and is now done (completed,
 * canceled, or failed), otherwise returns {@code false} to indicate stream is not yet started
 * or is in progress.
 */

public abstract boolean isDone();

/**
 * Highest stream priority. Passed to {@link android.net.http.BidirectionalStream.Builder#setPriority Builder#setPriority}.
 */

public static final int STREAM_PRIORITY_HIGHEST = 4; // 0x4

/**
 * Lowest stream priority. Passed to {@link android.net.http.BidirectionalStream.Builder#setPriority Builder#setPriority}.
 */

public static final int STREAM_PRIORITY_IDLE = 0; // 0x0

/**
 * Low stream priority. Passed to {@link android.net.http.BidirectionalStream.Builder#setPriority Builder#setPriority}.
 */

public static final int STREAM_PRIORITY_LOW = 2; // 0x2

/**
 * Very low stream priority. Passed to {@link android.net.http.BidirectionalStream.Builder#setPriority Builder#setPriority}.
 */

public static final int STREAM_PRIORITY_LOWEST = 1; // 0x1

/**
 * Medium stream priority. Passed to {@link android.net.http.BidirectionalStream.Builder#setPriority Builder#setPriority}. This is the
 * default priority given to the stream.
 */

public static final int STREAM_PRIORITY_MEDIUM = 3; // 0x3
/**
 * Builder for {@link android.net.http.BidirectionalStream BidirectionalStream}s. Allows configuring stream before constructing
 * it via {@link android.net.http.BidirectionalStream.Builder#build Builder#build}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract static class Builder {

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Sets the HTTP method for the request. Returns builder to facilitate chaining.
 *
 * @param method the method to use for request. Default is 'POST'
 * @return the builder to facilitate chaining
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream.Builder setHttpMethod(@android.annotation.NonNull java.lang.String method);

/**
 * Adds a request header. Returns builder to facilitate chaining.
 *
 * @param header the header name
 * @param value the header value
 * @return the builder to facilitate chaining
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream.Builder addHeader(@android.annotation.NonNull java.lang.String header, @android.annotation.NonNull java.lang.String value);

/**
 * Sets priority of the stream which should be one of the {@link #STREAM_PRIORITY_IDLE
 * STREAM_PRIORITY_*} values. The stream is given {@link #STREAM_PRIORITY_MEDIUM} priority
 * if this method is not called.
 *
 * @param priority priority of the stream which should be one of the {@link
 * #STREAM_PRIORITY_IDLE STREAM_PRIORITY_*} values.
 * Value is {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_IDLE}, {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_LOWEST}, {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_LOW}, {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_MEDIUM}, or {@link android.net.http.BidirectionalStream#STREAM_PRIORITY_HIGHEST}
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream.Builder setPriority(int priority);

/**
 * Sets whether to delay sending request headers until {@link android.net.http.BidirectionalStream#flush() BidirectionalStream#flush()}
 * is called. This flag is currently only respected when QUIC is negotiated.
 * When true, QUIC will send request header frame along with data frame(s)
 * as a single packet when possible.
 *
 * @param delayRequestHeadersUntilFirstFlush if true, sending request headers will be
 *         delayed
 * until flush() is called.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream.Builder setDelayRequestHeadersUntilFirstFlushEnabled(boolean delayRequestHeadersUntilFirstFlush);

/**
 * Sets {@link android.net.TrafficStats} tag to use when accounting socket traffic caused by
 * this request. See {@link android.net.TrafficStats} for more information. If no tag is
 * set (e.g. this method isn't called), then Android accounts for the socket traffic caused
 * by this request as if the tag value were set to 0.
 * <p>
 * <b>NOTE:</b>Setting a tag disallows sharing of sockets with requests
 * with other tags, which may adversely effect performance by prohibiting
 * connection sharing. In other words use of multiplexed sockets (e.g. HTTP/2
 * and QUIC) will only be allowed if all requests have the same socket tag.
 *
 * @param tag the tag value used to when accounting for socket traffic caused by this
 *            request. Tags between 0xFFFFFF00 and 0xFFFFFFFF are reserved and used
 *            internally by system services like {@link android.app.DownloadManager} when
 *            performing traffic on behalf of an application.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream.Builder setTrafficStatsTag(int tag);

/**
 * Sets specific UID to use when accounting socket traffic caused by this request. See
 * {@link android.net.TrafficStats} for more information. Designed for use when performing
 * an operation on behalf of another application. Caller must hold
 * {@code android.Manifest.permission#MODIFY_NETWORK_ACCOUNTING} permission. By default
 * traffic is attributed to UID of caller.
 * <p>
 * <b>NOTE:</b>Setting a UID disallows sharing of sockets with requests
 * with other UIDs, which may adversely effect performance by prohibiting
 * connection sharing. In other words use of multiplexed sockets (e.g. HTTP/2
 * and QUIC) will only be allowed if all requests have the same UID set.
 *
 * @param uid the UID to attribute socket traffic caused by this request.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream.Builder setTrafficStatsUid(int uid);

/**
 * Creates a {@link android.net.http.BidirectionalStream BidirectionalStream} using configuration from this {@link android.net.http.BidirectionalStream.Builder Builder}. The
 * returned {@code BidirectionalStream} can then be started by calling {@link android.net.http.BidirectionalStream#start  }.
 *
 * @return constructed {@link android.net.http.BidirectionalStream BidirectionalStream} using configuration from this {@link android.net.http.BidirectionalStream.Builder Builder}
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream build();
}

/**
 * Callback interface used to receive callbacks from a {@link android.net.http.BidirectionalStream BidirectionalStream}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface Callback {

/**
 * Invoked when the stream is ready for reading and writing. Consumer may call {@link android.net.http.BidirectionalStream#read read()} to start reading data. Consumer may call {@link android.net.http.BidirectionalStream#write write()} to start writing data.
 *
 * @param stream the stream that is ready. <strong>This is not guaranteed to be the same
 *        object as the one received by other callbacks, nor is it guaranteed to be the one
 *        returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.</strong> However, method
 *        calls on this object will have the same effects as calls on the original
 *        {@link android.net.http.BidirectionalStream BidirectionalStream}.
 */

public void onStreamReady(@android.annotation.NonNull android.net.http.BidirectionalStream stream);

/**
 * Invoked when initial response headers are received. Headers are available from {@code
 * info.}{@link android.net.http.UrlResponseInfo#getHeaders getHeaders()}. Consumer may call {@link android.net.http.BidirectionalStream#read read()} to start reading. Consumer may call {@link android.net.http.BidirectionalStream#write write()} to start writing or close the stream.
 *
 * @param stream the stream on which response headers were received. <strong>This is not
 *        guaranteed to be the same object as the one received by other callbacks, nor is
 *        it guaranteed to be the one returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.
 *        </strong> However, method calls on this object will have the same effects as
 *        calls on the original {@link android.net.http.BidirectionalStream BidirectionalStream}.
 * @param info the response information.
 */

public void onResponseHeadersReceived(@android.annotation.NonNull android.net.http.BidirectionalStream stream, @android.annotation.NonNull android.net.http.UrlResponseInfo info);

/**
 * Invoked when data is read into the buffer passed to {@link android.net.http.BidirectionalStream#read  read()}. Only part of the buffer may be populated. To continue reading, call {@link android.net.http.BidirectionalStream#read read()}. It may be invoked after {@code
 * onResponseTrailersReceived()}, if there was pending read data before trailers were
 * received.
 *
 * @param stream the stream on which the read completed. <strong>This is not guaranteed to
 *        be the same object as the one received by other callbacks, nor is it guaranteed
 *        to be the one returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.</strong>
 *        However, method calls on this object will have the same effects as calls on the
 *        original {@link android.net.http.BidirectionalStream BidirectionalStream}.
 * @param info the response information
 * @param buffer the buffer that was passed to {@link android.net.http.BidirectionalStream#read read()}, now
 * set to the end of the received data. If position is not updated, it means the remote side
 * has signaled that it will send no more data.
 * @param endOfStream if true, this is the last read data, remote will not send more data,
 *         and
 * the read side is closed.
 */

public void onReadCompleted(@android.annotation.NonNull android.net.http.BidirectionalStream stream, @android.annotation.NonNull android.net.http.UrlResponseInfo info, @android.annotation.NonNull java.nio.ByteBuffer buffer, boolean endOfStream);

/**
 * Invoked when the entire ByteBuffer passed to {@link android.net.http.BidirectionalStream#write write()} is
 * sent. The buffer's position is updated to be the same as the buffer's limit. The buffer's
 * limit is not changed. To continue writing, call {@link android.net.http.BidirectionalStream#write  write()}.
 *
 * @param stream the stream on which the write completed. <strong>This is not guaranteed to
 *        be the same object as the one received by other callbacks, nor is it guaranteed
 *        to be the one returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.</strong>
 *        However, method calls on this object will have the same effects as calls on the
 *        original {@link android.net.http.BidirectionalStream BidirectionalStream}.
 * @param info the response information
 * @param buffer the buffer that was passed to {@link android.net.http.BidirectionalStream#write write()}.
 *         The
 * buffer's position is set to the buffer's limit. The buffer's limit is not changed.
 * @param endOfStream the endOfStream flag that was passed to the corresponding {@link android.net.http.BidirectionalStream#write write()}. If true, the write side is closed.
 */

public void onWriteCompleted(@android.annotation.NonNull android.net.http.BidirectionalStream stream, @android.annotation.NonNull android.net.http.UrlResponseInfo info, @android.annotation.NonNull java.nio.ByteBuffer buffer, boolean endOfStream);

/**
 * Invoked when trailers are received before closing the stream. Only invoked when server
 * sends trailers, which it may not. May be invoked while there is read data remaining in
 * local buffer.
 *
 * Default implementation takes no action.
 *
 * @param stream the stream on which response trailers were received. <strong>This is not
 *        guaranteed to be the same object as the one received by other callbacks, nor is
 *        it guaranteed to be the one returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.
 *        </strong> However, method calls on this object will have the same effects as calls
 *        on the original {@link android.net.http.BidirectionalStream BidirectionalStream}.
 * @param info the response information
 * @param trailers the trailers received
 */

public void onResponseTrailersReceived(@android.annotation.NonNull android.net.http.BidirectionalStream stream, @android.annotation.NonNull android.net.http.UrlResponseInfo info, @android.annotation.NonNull android.net.http.HeaderBlock trailers);

/**
 * Invoked when there is no data to be read or written and the stream is closed successfully
 * remotely and locally. Once invoked, no further {@link android.net.http.BidirectionalStream.Callback BidirectionalStream.Callback}
 * methods will be invoked.
 *
 * @param stream the stream which is closed successfully. <strong>This is not guaranteed to
 *        be the same object as the one received by other callbacks, nor is it guaranteed
 *        to be the one returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.</strong>
 *        However, method calls on this object will have the same effects as calls on the
 *        original {@link android.net.http.BidirectionalStream BidirectionalStream}.
 * @param info the response information
 */

public void onSucceeded(@android.annotation.NonNull android.net.http.BidirectionalStream stream, @android.annotation.NonNull android.net.http.UrlResponseInfo info);

/**
 * Invoked if the stream failed for any reason after {@link android.net.http.BidirectionalStream#start BidirectionalStream#start}.
 * <a href="https://tools.ietf.org/html/rfc7540#section-7">HTTP/2 error codes</a> are
 * mapped to {@link android.net.http.NetworkException#getErrorCode NetworkException#getErrorCode} codes. Once invoked,
 * no further {@link android.net.http.BidirectionalStream.Callback BidirectionalStream.Callback} methods will be invoked.
 *
 * @param stream the stream which has failed. <strong>This is not guaranteed to
 *        be the same object as the one received by other callbacks, nor is it guaranteed
 *        to be the one returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.</strong>
 *        However, method calls on this object will have the same effects as calls on the
 *        original {@link android.net.http.BidirectionalStream BidirectionalStream}.
 * @param info the response information. May be {@code null} if no response was received.
 * @param error information about the failure
 */

public void onFailed(@android.annotation.NonNull android.net.http.BidirectionalStream stream, @android.annotation.Nullable android.net.http.UrlResponseInfo info, @android.annotation.NonNull android.net.http.HttpException error);

/**
 * Invoked if the stream was canceled via {@link android.net.http.BidirectionalStream#cancel BidirectionalStream#cancel}. Once invoked,
 * no further {@link android.net.http.BidirectionalStream.Callback BidirectionalStream.Callback} methods will be invoked. Default
 * implementation takes no action.
 *
 * @param stream the stream that was canceled. <strong>This is not guaranteed to
 *        be the same object as the one received by other callbacks, nor is it guaranteed
 *        to be the one returned by {@link android.net.http.BidirectionalStream.Builder#build BidirectionalStream.Builder#build}.</strong>
 *        However, method calls on this object will have the same effects as calls on the
 *        original {@link android.net.http.BidirectionalStream BidirectionalStream}.
 * @param info the response information. May be {@code null} if no response was received.
 */

public void onCanceled(@android.annotation.NonNull android.net.http.BidirectionalStream stream, @android.annotation.Nullable android.net.http.UrlResponseInfo info);
}

}

