/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.location.altitude;

import android.content.Context;
import android.location.Location;
import java.io.IOException;

/**
 * Converts altitudes reported above the World Geodetic System 1984 (WGS84) reference ellipsoid
 * into ones above Mean Sea Level.
 *
 * <p>Reference:
 *
 * <pre>
 * Brian Julian and Michael Angermann.
 * "Resource efficient and accurate altitude conversion to Mean Sea Level."
 * 2023 IEEE/ION Position, Location and Navigation Symposium (PLANS).
 * </pre>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class AltitudeConverter {

/**
 * Creates an instance that manages an independent cache to optimized conversions of locations
 * in proximity to one another.
 */

public AltitudeConverter() { throw new RuntimeException("Stub!"); }

/**
 * Adds a Mean Sea Level altitude to the {@code location}. In addition, adds a Mean Sea Level
 * altitude accuracy if the {@code location} has a finite and non-negative vertical accuracy;
 * otherwise, does not add a corresponding accuracy.
 *
 * <p>Must be called off the main thread as data may be loaded from raw assets.
 *
 * <br>
 * This method may take several seconds to complete, so it should
 * only be called from a worker thread.
 * @throws java.io.IOException              if an I/O error occurs when loading data from raw assets.
 * @throws java.lang.IllegalArgumentException if the {@code location} has an invalid latitude, longitude,
 *                                  or altitude above WGS84. Specifically, the latitude must be
 *                                  between -90 and 90 (both inclusive), the longitude must be
 *                                  between -180 and 180 (both inclusive), and the altitude
 *                                  above WGS84 must be finite.
 
 * @param context This value cannot be {@code null}.

 * @param location This value cannot be {@code null}.
 */

public void addMslAltitudeToLocation(@android.annotation.NonNull android.content.Context context, @android.annotation.NonNull android.location.Location location) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Same as {@link #addMslAltitudeToLocation(android.content.Context,android.location.Location)} except that this method can be
 * called on the main thread as data will not be loaded from raw assets. Returns true if a Mean
 * Sea Level altitude is added to the {@code location}; otherwise, returns false and leaves the
 * {@code location} unchanged.
 *
 * <p>Prior calls to {@link #addMslAltitudeToLocation(android.content.Context,android.location.Location)} off the main thread
 * are necessary to load data from raw assets. Example code on the main thread is as follows:
 *
 * <pre>{@code
 *   if (!mAltitudeConverter.tryAddMslAltitudeToLocation(location)) {
 *       // Queue up only one call off the main thread.
 *       if (mIsAltitudeConverterIdle) {
 *           mIsAltitudeConverterIdle = false;
 *           executeOffMainThread(() -> {
 *               try {
 *                   // Load raw assets for next call attempt on main thread.
 *                   mAltitudeConverter.addMslAltitudeToLocation(mContext, location);
 *               } catch (IOException e) {
 *                   Log.e(TAG, "Not loading raw assets: " + e);
 *               }
 *               mIsAltitudeConverterIdle = true;
 *           });
 *       }
 *   }
 * }</pre>

 * @param location This value cannot be {@code null}.
 */

public boolean tryAddMslAltitudeToLocation(@android.annotation.NonNull android.location.Location location) { throw new RuntimeException("Stub!"); }
}

