/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.crypto.hpke;

import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;

/**
 * Service Provider Interface for HPKE client API classes to communicate with implementations
 * of HPKE as described in RFC 9180.
 * <p>
 * There are no standard Java Cryptography Architecture names or interface classes for HPKE,
 * but instances of this class can be obtained by calling
 * {@code Provider.getService("ConscryptHpke", String SuiteName)} where {@code suiteName}
 * is the name of the HPKE suite, e.g.
 * {@code "DHKEM_X25519_HKDF_SHA256/HKDF_SHA256/AES_128_GCM"}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface HpkeSpi {

/**
 * Initialises an HPKE SPI in one of the sender modes described in RFC 9180.
 * <p>
 * If {@code psk} and {@code psk_id} are supplied then Pre-Shared Key Authentication
 * will be used.
 * <p>
 * If {@code senderKey} is supplied then Asymmetric Key Authentication will be used.
 * <p>
 * If neither is supplied then "base" mode (no sender authentication) will be used.
 * <p>
 * Note that only base mode is currently supported on Android.
 * <p>
 * Public and private keys must be supplied in a format that can be used by the
 * implementation.  An instance of the {@code "XDH"} {@link java.security.KeyFactory} can
 * be used to translate {@code KeySpecs} or keys from another {@link java.security.Provider}
 *
 * @param recipientKey public key of the recipient
 * @param info application-supplied information, may be null or empty
 * @param senderKey private key of the sender, for symmetric auth modes only, else null
 * @param psk pre-shared key, for PSK auth modes only, else null
 * @param psk_id pre-shared key ID, for PSK auth modes only, else null
 * @throws java.security.InvalidKeyException if recipientKey is null or an unsupported key format
 * @throws java.lang.UnsupportedOperationException if mode is not a supported HPKE mode
 * @throws java.lang.IllegalStateException if this SPI has already been initialised
 */

public void engineInitSender(@android.annotation.NonNull java.security.PublicKey recipientKey, @android.annotation.Nullable byte[] info, @android.annotation.Nullable java.security.PrivateKey senderKey, @android.annotation.Nullable byte[] psk, @android.annotation.Nullable byte[] psk_id) throws java.security.InvalidKeyException;

/**
 * Initialises an HPKE SPI in one of the sender modes described in RFC 9180 with
 * a predefined random seed to allow testing against known test vectors.
 * <p>
 * This mode provides absolutely no security and should only be used for testing
 * purposes.
 * <p>
 * If {@code psk} and {@code psk_id} are supplied then Pre-Shared Key Authentication
 * will be used.
 * <p>
 * If {@code senderKey} is supplied then Asymmetric Key Authentication will be used.
 * <p>
 * If neither is supplied then "base" mode (no sender authentication) will be used.
 * <p>
 * Note that only base mode is currently supported on Android.
 * <p>
 * Public and private keys must be supplied in a format that can be used by the
 * implementation.  An instance of the {@code "XDH"} {@link java.security.KeyFactory} can
 * be used to translate {@code KeySpecs} or keys from another {@link java.security.Provider}
 *
 *
 * @param recipientKey public key of the recipient
 * @param info application-supplied information, may be null or empty
 * @param senderKey private key of the sender, for symmetric auth modes only, else null
 * @param psk pre-shared key, for PSK auth modes only, else null
 * @param psk_id pre-shared key ID, for PSK auth modes only, else null
 * @param sKe Predetermined random seed, should only be usesd  for validation against
 *            known test vectors
 * @throws java.security.InvalidKeyException if recipientKey is null or an unsupported key format or senderKey
 *            is an unsupported key format
 * @throws java.lang.UnsupportedOperationException if mode is not a supported HPKE mode
 * @throws java.lang.IllegalStateException if this SPI has already been initialised
 */

public void engineInitSenderWithSeed(@android.annotation.NonNull java.security.PublicKey recipientKey, @android.annotation.Nullable byte[] info, @android.annotation.Nullable java.security.PrivateKey senderKey, @android.annotation.Nullable byte[] psk, @android.annotation.Nullable byte[] psk_id, @android.annotation.NonNull byte[] sKe) throws java.security.InvalidKeyException;

/**
 * Initialises an HPKE SPI in one of the sender modes described in RFC 9180.
 * <p>
 * If {@code psk} and {@code psk_id} are supplied then Pre-Shared Key Authentication
 * will be used.
 * <p>
 * If {@code senderKey} is supplied then Asymmetric Key Authentication will be used.
 * <p>
 * If neither is supplied then "base" mode (no sender authentication) will be used.
 * <p>
 * Note that only base mode is currently supported on Android.
 * <p>
 * Public and private keys must be supplied in a format that can be used by the
 * implementation.  An instance of the {@code "XDH"} {@link java.security.KeyFactory} can
 * be used to translate {@code KeySpecs} or keys from another {@link java.security.Provider}
 *
 * @param encapsulated encapsulated ephemeral key from a sender
 * @param recipientKey private key of the recipient
 * @param info application-supplied information, may be null or empty
 * @param senderKey public key of sender, for asymmetric auth modes only, else null
 * @param psk pre-shared key, for PSK auth modes only, else null
 * @param psk_id pre-shared key ID, for PSK auth modes only, else null
 * @throws java.security.InvalidKeyException if recipientKey is null or an unsupported key format or senderKey
 *         is an unsupported key format
 * @throws java.lang.UnsupportedOperationException if mode is not a supported HPKE mode
 * @throws java.lang.IllegalStateException if this SPI has already been initialised
 */

public void engineInitRecipient(@android.annotation.NonNull byte[] encapsulated, @android.annotation.NonNull java.security.PrivateKey recipientKey, @android.annotation.Nullable byte[] info, @android.annotation.Nullable java.security.PublicKey senderKey, @android.annotation.Nullable byte[] psk, @android.annotation.Nullable byte[] psk_id) throws java.security.InvalidKeyException;

/**
 * Seals a message, using the internal key schedule maintained by an HPKE sender SPI.
 *
 * @param plaintext the plaintext
 * @param aad optional associated data, may be null or empty
 * @return the ciphertext
 * @throws java.lang.NullPointerException if the plaintext is null
 * @throws java.lang.IllegalStateException if this SPI has not been initialised or if it was initialised
 *         as a recipient
 */

@android.annotation.NonNull
public byte[] engineSeal(@android.annotation.NonNull byte[] plaintext, @android.annotation.Nullable byte[] aad);

/**
 * Opens a message, using the internal key schedule maintained by an HPKE recipient SPI.
 *
 * @param ciphertext the ciphertext
 * @param aad optional associated data, may be null or empty
 * @return the plaintext
 * @throws java.lang.IllegalStateException if this SPI has not been initialised or if it was initialised
 *         as a sender
 * @throws java.security.GeneralSecurityException on decryption failures
 */

@android.annotation.NonNull
public byte[] engineOpen(@android.annotation.NonNull byte[] ciphertext, @android.annotation.Nullable byte[] aad) throws java.security.GeneralSecurityException;

/**
 * Exports secret key material from this SPI as described in RFC 9180.
 *
 * @param length  expected output length
 * @param context optional context string, may be null or empty
 * @return exported value
 * @throws java.lang.IllegalArgumentException if the length is not valid for the KDF in use
 * @throws java.lang.IllegalStateException if this SPI has not been initialised
 *
 */

@android.annotation.NonNull
public byte[] engineExport(int length, @android.annotation.Nullable byte[] context);

/**
 * Returns the encapsulated key material for an HPKE sender.
 *
 * @return the key material
 * @throws java.lang.IllegalStateException if this SPI has not been initialised or if it was initialised
 *         as a recipient
 */

@android.annotation.NonNull
public byte[] getEncapsulated();
}

